// ========================================
// PINGIG FINANCE - Standalone App
// Complete Band Financial Management
// ========================================

// ========== GLOBAL STATE ==========
let appData = {
    settings: {
        bandName: 'My Band',
        genre: '',
        formationYear: '',
        currency: 'USD',
        currencySymbol: '$',
        distanceUnit: 'km',
        dateFormat: 'YYYY-MM-DD',
        incomeTaxRate: 0,
        salesTaxRate: 0,
        taxYearStart: '01',
        vatRegistered: false,
        defaultFee: 0,
        mileageRate: 0,
        perDiem: 0,
        theme: 'dark',
        accentColor: 'orange',
        customIncomeCategories: [],
        customExpenseCategories: []
    },
    shows: [],
    crew: [],
    merchProducts: [],
    merchSales: [],
    merchOrders: [],
    streaming: [],
    studio: [],
    equipment: [],
    travel: [],
    marketing: [],
    otherIncome: [],
    otherExpenses: [],
    lastBackup: null
};

// Currency symbols map
const currencySymbols = {
    USD: '$', EUR: '€', GBP: '£', CAD: 'C$', AUD: 'A$',
    JPY: '¥', CHF: 'CHF', SEK: 'kr', NOK: 'kr', DKK: 'kr',
    PLN: 'zł', CZK: 'Kč', HUF: 'Ft', MXN: '$', BRL: 'R$',
    ARS: '$', CLP: '$', COP: '$', INR: '₹', CNY: '¥',
    KRW: '₩', NZD: 'NZ$', SGD: 'S$', HKD: 'HK$', ZAR: 'R',
    RUB: '₽', TRY: '₺', ILS: '₪', AED: 'د.إ', THB: '฿',
    PHP: '₱', IDR: 'Rp', MYR: 'RM'
};

// Current view state
let currentView = 'dashboard';
let currentDateRange = 'this-year';
let charts = {};

// ========== INITIALIZATION ==========
document.addEventListener('DOMContentLoaded', () => {
    loadData();
    initNavigation();
    initDateRange();
    initForms();
    updateDashboard();
    initCharts();
});

// Load data from localStorage
function loadData() {
    const saved = localStorage.getItem('pingigFinanceData');
    if (saved) {
        try {
            const parsed = JSON.parse(saved);
            appData = { ...appData, ...parsed };
        } catch (e) {
            console.error('Error loading data:', e);
        }
    }
    applySettings();
}

// Save data to localStorage
function saveData() {
    localStorage.setItem('pingigFinanceData', JSON.stringify(appData));
}

// Apply settings to UI
function applySettings() {
    // Update currency symbol
    appData.settings.currencySymbol = currencySymbols[appData.settings.currency] || '$';

    // Update sidebar
    document.getElementById('sidebar-band-name').textContent = appData.settings.bandName;
    document.getElementById('sidebar-currency').textContent = appData.settings.currency;

    // Update settings form
    document.getElementById('settings-band-name').value = appData.settings.bandName;
    document.getElementById('settings-genre').value = appData.settings.genre || '';
    document.getElementById('settings-year').value = appData.settings.formationYear || '';
    document.getElementById('settings-currency').value = appData.settings.currency;
    document.getElementById('settings-distance').value = appData.settings.distanceUnit;
    document.getElementById('settings-date-format').value = appData.settings.dateFormat;
    document.getElementById('settings-income-tax').value = appData.settings.incomeTaxRate || '';
    document.getElementById('settings-sales-tax').value = appData.settings.salesTaxRate || '';
    document.getElementById('settings-tax-year-start').value = appData.settings.taxYearStart;
    document.getElementById('settings-vat-registered').checked = appData.settings.vatRegistered;
    document.getElementById('settings-default-fee').value = appData.settings.defaultFee || '';
    document.getElementById('settings-mileage-rate').value = appData.settings.mileageRate || '';
    document.getElementById('settings-per-diem').value = appData.settings.perDiem || '';
    document.getElementById('settings-theme').value = appData.settings.theme;

    // Update last backup display
    if (appData.lastBackup) {
        document.getElementById('last-backup').textContent = `Last backup: ${formatDate(appData.lastBackup)}`;
    }
}

// ========== NAVIGATION ==========
function initNavigation() {
    document.querySelectorAll('.nav-item').forEach(item => {
        item.addEventListener('click', () => {
            const view = item.dataset.view;
            if (view) switchView(view);
        });
    });
}

function switchView(view) {
    // Update nav
    document.querySelectorAll('.nav-item').forEach(item => {
        item.classList.toggle('active', item.dataset.view === view);
    });

    // Update views
    document.querySelectorAll('.view').forEach(v => {
        v.classList.toggle('active', v.id === `${view}-view`);
    });

    // Update page title
    const titles = {
        dashboard: ['Dashboard', 'Overview of your band\'s finances'],
        shows: ['Shows & Gigs', 'Track your live performances'],
        'merch-sales': ['Merch Sales', 'Track merchandise revenue'],
        streaming: ['Streaming & Royalties', 'Track digital income'],
        'other-income': ['Other Income', 'Additional revenue streams'],
        crew: ['Crew & Team', 'Manage your team and payments'],
        'merch-costs': ['Merch Production', 'Track merchandise costs'],
        studio: ['Studio & Recording', 'Track recording expenses'],
        equipment: ['Equipment & Gear', 'Manage your equipment'],
        travel: ['Travel & Transport', 'Track travel expenses'],
        marketing: ['Marketing & Promo', 'Track promotional costs'],
        'other-expenses': ['Other Expenses', 'Miscellaneous costs'],
        reports: ['Financial Reports', 'Detailed financial analysis'],
        tax: ['Tax Summary', 'Tax calculation and export'],
        settings: ['Settings', 'Configure your preferences'],
        backup: ['Backup & Export', 'Manage your data']
    };

    const [title, subtitle] = titles[view] || ['', ''];
    document.getElementById('current-page-title').textContent = title;
    document.getElementById('current-page-subtitle').textContent = subtitle;

    currentView = view;

    // Load view-specific data
    loadViewData(view);
}

function loadViewData(view) {
    switch(view) {
        case 'dashboard':
            updateDashboard();
            break;
        case 'shows':
            renderShows();
            break;
        case 'crew':
            renderCrew();
            break;
        case 'merch-sales':
            renderMerchSales();
            break;
        case 'merch-costs':
            renderMerchCosts();
            break;
        case 'streaming':
            renderStreaming();
            break;
        case 'studio':
            renderStudio();
            break;
        case 'equipment':
            renderEquipment();
            break;
        case 'travel':
            renderTravel();
            break;
        case 'marketing':
            renderMarketing();
            break;
        case 'other-income':
            renderOtherIncome();
            break;
        case 'other-expenses':
            renderOtherExpenses();
            break;
        case 'reports':
            generateReport();
            break;
        case 'tax':
            updateTaxSummary();
            break;
    }
}

// ========== DATE RANGE ==========
function initDateRange() {
    document.getElementById('date-range').addEventListener('change', (e) => {
        currentDateRange = e.target.value;
        loadViewData(currentView);
    });
}

function getDateRange() {
    const now = new Date();
    let start, end;

    switch(currentDateRange) {
        case 'this-month':
            start = new Date(now.getFullYear(), now.getMonth(), 1);
            end = new Date(now.getFullYear(), now.getMonth() + 1, 0);
            break;
        case 'last-month':
            start = new Date(now.getFullYear(), now.getMonth() - 1, 1);
            end = new Date(now.getFullYear(), now.getMonth(), 0);
            break;
        case 'this-quarter':
            const quarter = Math.floor(now.getMonth() / 3);
            start = new Date(now.getFullYear(), quarter * 3, 1);
            end = new Date(now.getFullYear(), (quarter + 1) * 3, 0);
            break;
        case 'this-year':
            start = new Date(now.getFullYear(), 0, 1);
            end = new Date(now.getFullYear(), 11, 31);
            break;
        case 'all-time':
            start = new Date(2000, 0, 1);
            end = new Date(2100, 11, 31);
            break;
        default:
            start = new Date(now.getFullYear(), 0, 1);
            end = new Date(now.getFullYear(), 11, 31);
    }

    return { start, end };
}

function isInDateRange(dateStr) {
    const { start, end } = getDateRange();
    const date = new Date(dateStr);
    return date >= start && date <= end;
}

// ========== FORMATTING ==========
function formatCurrency(amount) {
    const num = parseFloat(amount) || 0;
    return `${appData.settings.currencySymbol}${num.toLocaleString(undefined, { minimumFractionDigits: 2, maximumFractionDigits: 2 })}`;
}

function formatDate(dateStr) {
    if (!dateStr) return '-';
    const date = new Date(dateStr);
    const format = appData.settings.dateFormat;

    const day = date.getDate().toString().padStart(2, '0');
    const month = (date.getMonth() + 1).toString().padStart(2, '0');
    const year = date.getFullYear();

    switch(format) {
        case 'MM/DD/YYYY': return `${month}/${day}/${year}`;
        case 'DD/MM/YYYY': return `${day}/${month}/${year}`;
        default: return `${year}-${month}-${day}`;
    }
}

function formatDistance(distance) {
    const unit = appData.settings.distanceUnit;
    return `${parseFloat(distance).toLocaleString()} ${unit}`;
}

// ========== DASHBOARD ==========
function updateDashboard() {
    const { start, end } = getDateRange();

    // Calculate totals
    let totalIncome = 0;
    let totalExpenses = 0;

    // Shows income
    appData.shows.filter(s => isInDateRange(s.date)).forEach(show => {
        totalIncome += parseFloat(show.fee) || 0;
        totalIncome += parseFloat(show.merch) || 0;
        totalIncome += parseFloat(show.door) || 0;
        totalExpenses += parseFloat(show.travelCost) || 0;
        totalExpenses += parseFloat(show.accommodation) || 0;
        totalExpenses += parseFloat(show.crewCost) || 0;
        totalExpenses += parseFloat(show.otherCost) || 0;
    });

    // Merch sales income
    appData.merchSales.filter(s => isInDateRange(s.date)).forEach(sale => {
        totalIncome += parseFloat(sale.total) || 0;
    });

    // Streaming income
    appData.streaming.filter(s => isInDateRange(s.date)).forEach(s => {
        totalIncome += parseFloat(s.amount) || 0;
    });

    // Other income
    appData.otherIncome.filter(i => isInDateRange(i.date)).forEach(i => {
        totalIncome += parseFloat(i.amount) || 0;
    });

    // Crew expenses
    appData.crew.forEach(member => {
        if (member.paymentType !== 'unpaid') {
            // Calculate based on shows in range
            const showsInRange = appData.shows.filter(s => isInDateRange(s.date)).length;
            if (member.paymentType === 'per-show') {
                totalExpenses += (parseFloat(member.rate) || 0) * showsInRange;
            }
        }
    });

    // Merch production costs
    appData.merchOrders.filter(o => isInDateRange(o.date)).forEach(order => {
        totalExpenses += parseFloat(order.totalCost) || 0;
    });

    // Studio costs
    appData.studio.filter(s => isInDateRange(s.date)).forEach(s => {
        totalExpenses += parseFloat(s.total) || 0;
    });

    // Equipment costs
    appData.equipment.filter(e => isInDateRange(e.date) && e.type !== 'sale').forEach(e => {
        totalExpenses += parseFloat(e.cost) || 0;
    });

    // Travel costs
    appData.travel.filter(t => isInDateRange(t.date)).forEach(t => {
        totalExpenses += parseFloat(t.cost) || 0;
    });

    // Marketing costs
    appData.marketing.filter(m => isInDateRange(m.date)).forEach(m => {
        totalExpenses += parseFloat(m.cost) || 0;
    });

    // Other expenses
    appData.otherExpenses.filter(e => isInDateRange(e.date)).forEach(e => {
        totalExpenses += parseFloat(e.amount) || 0;
    });

    const netProfit = totalIncome - totalExpenses;
    const showsCount = appData.shows.filter(s => isInDateRange(s.date)).length;
    const avgPerShow = showsCount > 0 ? netProfit / showsCount : 0;

    // Update cards
    document.getElementById('total-income').textContent = formatCurrency(totalIncome);
    document.getElementById('total-expenses').textContent = formatCurrency(totalExpenses);
    document.getElementById('net-profit').textContent = formatCurrency(netProfit);
    document.getElementById('shows-count').textContent = showsCount;
    document.getElementById('avg-per-show').textContent = `Avg: ${formatCurrency(avgPerShow)}/show`;

    // Update profit trend styling
    const profitTrend = document.getElementById('profit-trend');
    if (netProfit >= 0) {
        profitTrend.className = 'card-trend positive';
        profitTrend.textContent = '+' + ((netProfit / Math.max(totalIncome, 1)) * 100).toFixed(0) + '%';
    } else {
        profitTrend.className = 'card-trend negative';
        profitTrend.textContent = ((netProfit / Math.max(totalExpenses, 1)) * 100).toFixed(0) + '%';
    }

    // Quick stats
    document.getElementById('crew-count').textContent = appData.crew.length;
    document.getElementById('merch-items').textContent = appData.merchProducts.length;
    document.getElementById('studio-hours').textContent = appData.studio.reduce((sum, s) => sum + (parseFloat(s.hours) || 0), 0);
    document.getElementById('travel-distance').textContent = formatDistance(
        appData.travel.filter(t => isInDateRange(t.date)).reduce((sum, t) => sum + (parseFloat(t.distance) || 0), 0)
    );

    // Recent transactions
    renderRecentTransactions();
}

function renderRecentTransactions() {
    const container = document.getElementById('recent-transactions');
    const transactions = [];

    // Gather recent transactions
    appData.shows.forEach(show => {
        if (show.fee) {
            transactions.push({
                type: 'income',
                icon: '🎤',
                title: `Show at ${show.venue}`,
                details: show.city,
                amount: parseFloat(show.fee) + (parseFloat(show.merch) || 0) + (parseFloat(show.door) || 0),
                date: show.date
            });
        }
    });

    appData.streaming.forEach(s => {
        transactions.push({
            type: 'income',
            icon: '🎧',
            title: `${s.platform} Payment`,
            details: s.period || '',
            amount: parseFloat(s.amount),
            date: s.date
        });
    });

    appData.studio.forEach(s => {
        transactions.push({
            type: 'expense',
            icon: '🎚️',
            title: `Studio: ${s.name || s.type}`,
            details: `${s.hours} hours`,
            amount: parseFloat(s.total),
            date: s.date
        });
    });

    appData.equipment.filter(e => e.type !== 'sale').forEach(e => {
        transactions.push({
            type: 'expense',
            icon: '🎸',
            title: e.name,
            details: e.category,
            amount: parseFloat(e.cost),
            date: e.date
        });
    });

    // Sort by date and take recent 5
    transactions.sort((a, b) => new Date(b.date) - new Date(a.date));
    const recent = transactions.slice(0, 5);

    if (recent.length === 0) {
        container.innerHTML = `
            <div class="empty-state">
                <div class="empty-icon">📋</div>
                <h3>No transactions yet</h3>
                <p>Start by adding a show or expense</p>
            </div>
        `;
        return;
    }

    container.innerHTML = recent.map(t => `
        <div class="activity-item">
            <div class="activity-icon ${t.type}">${t.icon}</div>
            <div class="activity-info">
                <div class="activity-title">${t.title}</div>
                <div class="activity-details">${t.details}</div>
            </div>
            <div class="activity-amount ${t.type}">${t.type === 'income' ? '+' : '-'}${formatCurrency(t.amount)}</div>
            <div class="activity-date">${formatDate(t.date)}</div>
        </div>
    `).join('');
}

// ========== CHARTS ==========
function initCharts() {
    // Income vs Expenses Chart
    const ctx1 = document.getElementById('income-expenses-chart');
    if (ctx1) {
        charts.incomeExpenses = new Chart(ctx1, {
            type: 'line',
            data: {
                labels: getMonthLabels(),
                datasets: [
                    {
                        label: 'Income',
                        data: getMonthlyIncome(),
                        borderColor: '#ff6b00',
                        backgroundColor: 'rgba(255, 107, 0, 0.1)',
                        fill: true,
                        tension: 0.4
                    },
                    {
                        label: 'Expenses',
                        data: getMonthlyExpenses(),
                        borderColor: '#ff1744',
                        backgroundColor: 'rgba(255, 23, 68, 0.1)',
                        fill: true,
                        tension: 0.4
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: { display: false }
                },
                scales: {
                    x: {
                        grid: { color: 'rgba(255,255,255,0.05)' },
                        ticks: { color: '#8a9299' }
                    },
                    y: {
                        grid: { color: 'rgba(255,255,255,0.05)' },
                        ticks: { color: '#8a9299' }
                    }
                }
            }
        });
    }

    // Expense Breakdown Chart
    const ctx2 = document.getElementById('expense-breakdown-chart');
    if (ctx2) {
        charts.expenseBreakdown = new Chart(ctx2, {
            type: 'doughnut',
            data: {
                labels: ['Crew', 'Studio', 'Equipment', 'Travel', 'Marketing', 'Other'],
                datasets: [{
                    data: getExpenseBreakdown(),
                    backgroundColor: [
                        '#ff6b00',
                        '#b537f2',
                        '#00d9ff',
                        '#00e676',
                        '#ffc107',
                        '#8a9299'
                    ]
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'right',
                        labels: { color: '#8a9299' }
                    }
                }
            }
        });
    }
}

function getMonthLabels() {
    const months = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];
    const year = new Date().getFullYear();
    return months.map(m => `${m} ${year}`);
}

function getMonthlyIncome() {
    const monthly = new Array(12).fill(0);
    const year = new Date().getFullYear();

    appData.shows.forEach(show => {
        const date = new Date(show.date);
        if (date.getFullYear() === year) {
            monthly[date.getMonth()] += (parseFloat(show.fee) || 0) + (parseFloat(show.merch) || 0);
        }
    });

    appData.streaming.forEach(s => {
        const date = new Date(s.date);
        if (date.getFullYear() === year) {
            monthly[date.getMonth()] += parseFloat(s.amount) || 0;
        }
    });

    return monthly;
}

function getMonthlyExpenses() {
    const monthly = new Array(12).fill(0);
    const year = new Date().getFullYear();

    appData.studio.forEach(s => {
        const date = new Date(s.date);
        if (date.getFullYear() === year) {
            monthly[date.getMonth()] += parseFloat(s.total) || 0;
        }
    });

    appData.travel.forEach(t => {
        const date = new Date(t.date);
        if (date.getFullYear() === year) {
            monthly[date.getMonth()] += parseFloat(t.cost) || 0;
        }
    });

    appData.equipment.forEach(e => {
        const date = new Date(e.date);
        if (date.getFullYear() === year && e.type !== 'sale') {
            monthly[date.getMonth()] += parseFloat(e.cost) || 0;
        }
    });

    return monthly;
}

function getExpenseBreakdown() {
    const { start, end } = getDateRange();

    let crew = 0, studio = 0, equipment = 0, travel = 0, marketing = 0, other = 0;

    appData.studio.filter(s => isInDateRange(s.date)).forEach(s => {
        studio += parseFloat(s.total) || 0;
    });

    appData.equipment.filter(e => isInDateRange(e.date) && e.type !== 'sale').forEach(e => {
        equipment += parseFloat(e.cost) || 0;
    });

    appData.travel.filter(t => isInDateRange(t.date)).forEach(t => {
        travel += parseFloat(t.cost) || 0;
    });

    appData.marketing.filter(m => isInDateRange(m.date)).forEach(m => {
        marketing += parseFloat(m.cost) || 0;
    });

    appData.otherExpenses.filter(e => isInDateRange(e.date)).forEach(e => {
        other += parseFloat(e.amount) || 0;
    });

    // Crew costs from shows
    appData.shows.filter(s => isInDateRange(s.date)).forEach(show => {
        crew += parseFloat(show.crewCost) || 0;
    });

    return [crew, studio, equipment, travel, marketing, other];
}

function updateCharts() {
    if (charts.incomeExpenses) {
        charts.incomeExpenses.data.datasets[0].data = getMonthlyIncome();
        charts.incomeExpenses.data.datasets[1].data = getMonthlyExpenses();
        charts.incomeExpenses.update();
    }

    if (charts.expenseBreakdown) {
        charts.expenseBreakdown.data.datasets[0].data = getExpenseBreakdown();
        charts.expenseBreakdown.update();
    }
}

// ========== SHOWS ==========
function renderShows() {
    const container = document.getElementById('shows-list');
    const shows = appData.shows.filter(s => isInDateRange(s.date)).sort((a, b) => new Date(b.date) - new Date(a.date));

    // Update stats
    const totalIncome = shows.reduce((sum, s) => sum + (parseFloat(s.fee) || 0) + (parseFloat(s.merch) || 0) + (parseFloat(s.door) || 0), 0);
    const avgFee = shows.length > 0 ? shows.reduce((sum, s) => sum + (parseFloat(s.fee) || 0), 0) / shows.length : 0;
    const avgMerch = shows.length > 0 ? shows.reduce((sum, s) => sum + (parseFloat(s.merch) || 0), 0) / shows.length : 0;
    const totalCosts = shows.reduce((sum, s) => sum + (parseFloat(s.travelCost) || 0) + (parseFloat(s.accommodation) || 0) + (parseFloat(s.crewCost) || 0) + (parseFloat(s.otherCost) || 0), 0);
    const profitMargin = totalIncome > 0 ? ((totalIncome - totalCosts) / totalIncome * 100) : 0;

    document.getElementById('shows-total-income').textContent = formatCurrency(totalIncome);
    document.getElementById('shows-avg-fee').textContent = formatCurrency(avgFee);
    document.getElementById('shows-merch-avg').textContent = formatCurrency(avgMerch);
    document.getElementById('shows-profit-margin').textContent = `${profitMargin.toFixed(0)}%`;

    if (shows.length === 0) {
        container.innerHTML = `
            <div class="empty-state">
                <div class="empty-icon">🎤</div>
                <h3>No shows yet</h3>
                <p>Add your first show to start tracking</p>
                <button onclick="openModal('show')">+ Add Show</button>
            </div>
        `;
        return;
    }

    container.innerHTML = shows.map(show => {
        const income = (parseFloat(show.fee) || 0) + (parseFloat(show.merch) || 0) + (parseFloat(show.door) || 0);
        const costs = (parseFloat(show.travelCost) || 0) + (parseFloat(show.accommodation) || 0) + (parseFloat(show.crewCost) || 0) + (parseFloat(show.otherCost) || 0);
        const profit = income - costs;

        return `
            <div class="data-item">
                <div class="data-icon">🎤</div>
                <div class="data-main">
                    <div class="data-title">${show.venue}</div>
                    <div class="data-subtitle">${show.city} • ${show.type || 'Show'}</div>
                </div>
                <div class="data-meta">
                    <div class="data-amount ${profit >= 0 ? 'income' : 'expense'}">${profit >= 0 ? '+' : ''}${formatCurrency(profit)}</div>
                    <div class="data-date">${formatDate(show.date)}</div>
                </div>
                <div class="data-actions">
                    <button class="data-btn" onclick="editShow('${show.id}')">✏️</button>
                    <button class="data-btn delete" onclick="confirmDelete('show', '${show.id}')">🗑️</button>
                </div>
            </div>
        `;
    }).join('');
}

function saveShow(e) {
    e.preventDefault();

    const id = document.getElementById('show-id').value || generateId();
    const show = {
        id,
        venue: document.getElementById('show-venue').value,
        city: document.getElementById('show-city').value,
        date: document.getElementById('show-date').value,
        type: document.getElementById('show-type').value,
        fee: document.getElementById('show-fee').value,
        feeStatus: document.getElementById('show-fee-status').value,
        merch: document.getElementById('show-merch').value,
        door: document.getElementById('show-door').value,
        travelCost: document.getElementById('show-travel-cost').value,
        accommodation: document.getElementById('show-accommodation').value,
        crewCost: document.getElementById('show-crew-cost').value,
        otherCost: document.getElementById('show-other-cost').value,
        attendance: document.getElementById('show-attendance').value,
        capacity: document.getElementById('show-capacity').value,
        notes: document.getElementById('show-notes').value
    };

    const existingIndex = appData.shows.findIndex(s => s.id === id);
    if (existingIndex >= 0) {
        appData.shows[existingIndex] = show;
    } else {
        appData.shows.push(show);
    }

    saveData();
    closeModal('show');
    renderShows();
    updateDashboard();
    updateCharts();
    showToast('Show saved successfully!');
}

function editShow(id) {
    const show = appData.shows.find(s => s.id === id);
    if (!show) return;

    document.getElementById('show-modal-title').textContent = 'Edit Show';
    document.getElementById('show-id').value = show.id;
    document.getElementById('show-venue').value = show.venue;
    document.getElementById('show-city').value = show.city;
    document.getElementById('show-date').value = show.date;
    document.getElementById('show-type').value = show.type || 'headline';
    document.getElementById('show-fee').value = show.fee || '';
    document.getElementById('show-fee-status').value = show.feeStatus || 'pending';
    document.getElementById('show-merch').value = show.merch || '';
    document.getElementById('show-door').value = show.door || '';
    document.getElementById('show-travel-cost').value = show.travelCost || '';
    document.getElementById('show-accommodation').value = show.accommodation || '';
    document.getElementById('show-crew-cost').value = show.crewCost || '';
    document.getElementById('show-other-cost').value = show.otherCost || '';
    document.getElementById('show-attendance').value = show.attendance || '';
    document.getElementById('show-capacity').value = show.capacity || '';
    document.getElementById('show-notes').value = show.notes || '';

    openModal('show');
}

function deleteShow(id) {
    appData.shows = appData.shows.filter(s => s.id !== id);
    saveData();
    renderShows();
    updateDashboard();
    updateCharts();
    showToast('Show deleted');
}

// ========== CREW ==========
function renderCrew() {
    const container = document.getElementById('crew-list');
    const crew = appData.crew;

    // Update stats
    const bandCount = crew.filter(c => c.category === 'band').length;
    const crewCount = crew.filter(c => c.category === 'crew').length;
    const externalCount = crew.filter(c => c.category === 'external').length;

    // Calculate total paid
    let totalPaid = 0;
    const showsInRange = appData.shows.filter(s => isInDateRange(s.date)).length;
    crew.forEach(member => {
        if (member.paymentType === 'per-show') {
            totalPaid += (parseFloat(member.rate) || 0) * showsInRange;
        }
    });

    document.getElementById('crew-total-paid').textContent = formatCurrency(totalPaid);
    document.getElementById('crew-band-count').textContent = bandCount;
    document.getElementById('crew-crew-count').textContent = crewCount;
    document.getElementById('crew-external-count').textContent = externalCount;

    if (crew.length === 0) {
        container.innerHTML = `
            <div class="empty-state">
                <div class="empty-icon">👥</div>
                <h3>No team members yet</h3>
                <p>Add band members and crew</p>
                <button onclick="openModal('crew')">+ Add Person</button>
            </div>
        `;
        return;
    }

    container.innerHTML = crew.map(member => {
        const initials = member.name.split(' ').map(n => n[0]).join('').substring(0, 2).toUpperCase();
        const paymentText = member.paymentType === 'unpaid' ? 'Volunteer' :
            member.paymentType === 'per-show' ? `${formatCurrency(member.rate)}/show` :
            member.paymentType === 'daily' ? `${formatCurrency(member.rate)}/day` :
            member.paymentType === 'percentage' ? `${member.rate}%` :
            formatCurrency(member.rate);

        return `
            <div class="crew-card">
                <div class="crew-header">
                    <div class="crew-avatar ${member.category}">${initials}</div>
                    <div class="crew-info">
                        <h4>${member.name}</h4>
                        <span class="role">${member.role}</span>
                    </div>
                    <span class="crew-category-badge ${member.category}">${member.category}</span>
                </div>
                <div class="crew-payment">
                    <span class="payment-type">${member.paymentType.replace('-', ' ')}</span>
                    <span class="payment-rate">${paymentText}</span>
                </div>
                <div class="crew-actions">
                    <button class="crew-edit-btn" onclick="editCrew('${member.id}')">Edit</button>
                    <button class="crew-delete-btn" onclick="confirmDelete('crew', '${member.id}')">Delete</button>
                </div>
            </div>
        `;
    }).join('');
}

function saveCrew(e) {
    e.preventDefault();

    const id = document.getElementById('crew-id').value || generateId();
    const member = {
        id,
        name: document.getElementById('crew-name').value,
        category: document.getElementById('crew-category').value,
        role: document.getElementById('crew-role').value,
        paymentType: document.getElementById('crew-payment-type').value,
        rate: document.getElementById('crew-rate').value,
        email: document.getElementById('crew-email').value,
        phone: document.getElementById('crew-phone').value,
        notes: document.getElementById('crew-notes').value
    };

    const existingIndex = appData.crew.findIndex(c => c.id === id);
    if (existingIndex >= 0) {
        appData.crew[existingIndex] = member;
    } else {
        appData.crew.push(member);
    }

    saveData();
    closeModal('crew');
    renderCrew();
    updateDashboard();
    showToast('Crew member saved!');
}

function editCrew(id) {
    const member = appData.crew.find(c => c.id === id);
    if (!member) return;

    document.getElementById('crew-modal-title').textContent = 'Edit Crew Member';
    document.getElementById('crew-id').value = member.id;
    document.getElementById('crew-name').value = member.name;
    document.getElementById('crew-category').value = member.category;
    document.getElementById('crew-role').value = member.role;
    document.getElementById('crew-payment-type').value = member.paymentType;
    document.getElementById('crew-rate').value = member.rate || '';
    document.getElementById('crew-email').value = member.email || '';
    document.getElementById('crew-phone').value = member.phone || '';
    document.getElementById('crew-notes').value = member.notes || '';

    openModal('crew');
}

function deleteCrew(id) {
    appData.crew = appData.crew.filter(c => c.id !== id);
    saveData();
    renderCrew();
    updateDashboard();
    showToast('Crew member deleted');
}

// ========== MERCH SALES ==========
function renderMerchSales() {
    const container = document.getElementById('merch-sales-list');
    const sales = appData.merchSales.filter(s => isInDateRange(s.date)).sort((a, b) => new Date(b.date) - new Date(a.date));

    // Update stats
    const totalRevenue = sales.reduce((sum, s) => sum + (parseFloat(s.total) || 0), 0);
    const itemsSold = sales.reduce((sum, s) => sum + (parseInt(s.qty) || 0), 0);

    // Calculate profit (revenue - cost)
    let totalProfit = 0;
    sales.forEach(sale => {
        const product = appData.merchProducts.find(p => p.id === sale.productId);
        if (product) {
            const costPer = parseFloat(product.costPer) || 0;
            totalProfit += (parseFloat(sale.total) || 0) - (costPer * (parseInt(sale.qty) || 0));
        } else {
            totalProfit += parseFloat(sale.total) || 0;
        }
    });

    // Find best seller
    const productSales = {};
    sales.forEach(s => {
        if (!productSales[s.productId]) productSales[s.productId] = 0;
        productSales[s.productId] += parseInt(s.qty) || 0;
    });
    let bestSeller = '-';
    let maxSold = 0;
    Object.entries(productSales).forEach(([id, qty]) => {
        if (qty > maxSold) {
            maxSold = qty;
            const product = appData.merchProducts.find(p => p.id === id);
            bestSeller = product ? product.name : '-';
        }
    });

    document.getElementById('merch-total-revenue').textContent = formatCurrency(totalRevenue);
    document.getElementById('merch-total-profit').textContent = formatCurrency(totalProfit);
    document.getElementById('merch-items-sold').textContent = itemsSold;
    document.getElementById('merch-best-seller').textContent = bestSeller;

    if (sales.length === 0) {
        container.innerHTML = `
            <div class="empty-state">
                <div class="empty-icon">💰</div>
                <h3>No sales logged yet</h3>
                <p>Log your merch sales to track revenue</p>
                <button onclick="openModal('merch-sale')">+ Log Sale</button>
            </div>
        `;
        return;
    }

    container.innerHTML = sales.map(sale => {
        const product = appData.merchProducts.find(p => p.id === sale.productId);
        const productName = product ? product.name : sale.product || 'Unknown';

        return `
            <div class="data-item">
                <div class="data-icon">💰</div>
                <div class="data-main">
                    <div class="data-title">${productName}</div>
                    <div class="data-subtitle">${sale.qty}x @ ${formatCurrency(sale.price)} • ${sale.location}</div>
                </div>
                <div class="data-meta">
                    <div class="data-amount income">+${formatCurrency(sale.total)}</div>
                    <div class="data-date">${formatDate(sale.date)}</div>
                </div>
                <div class="data-actions">
                    <button class="data-btn delete" onclick="confirmDelete('merch-sale', '${sale.id}')">🗑️</button>
                </div>
            </div>
        `;
    }).join('');
}

function saveMerchSale(e) {
    e.preventDefault();

    const id = document.getElementById('merch-sale-id').value || generateId();
    const sale = {
        id,
        date: document.getElementById('merch-sale-date').value,
        location: document.getElementById('merch-sale-location').value,
        productId: document.getElementById('merch-sale-product').value,
        qty: document.getElementById('merch-sale-qty').value,
        price: document.getElementById('merch-sale-price').value,
        total: document.getElementById('merch-sale-total').value,
        showId: document.getElementById('merch-sale-show').value,
        notes: document.getElementById('merch-sale-notes').value
    };

    appData.merchSales.push(sale);
    saveData();
    closeModal('merch-sale');
    renderMerchSales();
    updateDashboard();
    showToast('Sale logged!');
}

// ========== MERCH PRODUCTION ==========
function renderMerchCosts() {
    const container = document.getElementById('merch-costs-list');
    const orders = appData.merchOrders.filter(o => isInDateRange(o.date)).sort((a, b) => new Date(b.date) - new Date(a.date));

    // Update stats
    const totalCost = orders.reduce((sum, o) => sum + (parseFloat(o.totalCost) || 0), 0);
    const productCount = appData.merchProducts.length;

    // Calculate average margin
    let margins = [];
    appData.merchProducts.forEach(p => {
        if (p.sellPrice && p.costPer) {
            const margin = ((p.sellPrice - p.costPer) / p.sellPrice) * 100;
            margins.push(margin);
        }
    });
    const avgMargin = margins.length > 0 ? margins.reduce((a, b) => a + b, 0) / margins.length : 0;

    document.getElementById('merch-production-cost').textContent = formatCurrency(totalCost);
    document.getElementById('merch-avg-margin').textContent = `${avgMargin.toFixed(0)}%`;
    document.getElementById('merch-product-count').textContent = productCount;

    if (orders.length === 0) {
        container.innerHTML = `
            <div class="empty-state">
                <div class="empty-icon">👕</div>
                <h3>No production orders yet</h3>
                <p>Add your merch production orders</p>
                <button onclick="openModal('merch-order')">+ Add Order</button>
            </div>
        `;
        return;
    }

    container.innerHTML = orders.map(order => `
        <div class="data-item">
            <div class="data-icon">👕</div>
            <div class="data-main">
                <div class="data-title">${order.name}</div>
                <div class="data-subtitle">${order.qty}x @ ${formatCurrency(order.costPer)} • ${order.supplier || 'Unknown supplier'}</div>
            </div>
            <div class="data-meta">
                <div class="data-amount expense">-${formatCurrency(order.totalCost)}</div>
                <div class="data-date">${formatDate(order.date)}</div>
            </div>
            <div class="data-actions">
                <button class="data-btn delete" onclick="confirmDelete('merch-order', '${order.id}')">🗑️</button>
            </div>
        </div>
    `).join('');
}

function saveMerchOrder(e) {
    e.preventDefault();

    const id = document.getElementById('merch-order-id').value || generateId();
    const order = {
        id,
        name: document.getElementById('merch-product-name').value,
        category: document.getElementById('merch-product-category').value,
        date: document.getElementById('merch-order-date').value,
        supplier: document.getElementById('merch-supplier').value,
        qty: document.getElementById('merch-order-qty').value,
        costPer: document.getElementById('merch-cost-per').value,
        totalCost: document.getElementById('merch-total-cost').value,
        sellPrice: document.getElementById('merch-sell-price').value,
        notes: document.getElementById('merch-order-notes').value
    };

    appData.merchOrders.push(order);

    // Also add/update product
    const existingProduct = appData.merchProducts.find(p => p.name.toLowerCase() === order.name.toLowerCase());
    if (existingProduct) {
        existingProduct.costPer = order.costPer;
        existingProduct.sellPrice = order.sellPrice;
    } else {
        appData.merchProducts.push({
            id: generateId(),
            name: order.name,
            category: order.category,
            costPer: order.costPer,
            sellPrice: order.sellPrice
        });
    }

    saveData();
    closeModal('merch-order');
    renderMerchCosts();
    updateDashboard();
    updateCharts();
    showToast('Production order saved!');
}

// ========== STREAMING ==========
function renderStreaming() {
    const container = document.getElementById('streaming-list');
    const payments = appData.streaming.filter(s => isInDateRange(s.date)).sort((a, b) => new Date(b.date) - new Date(a.date));

    const total = payments.reduce((sum, p) => sum + (parseFloat(p.amount) || 0), 0);
    const monthlyAvg = payments.length > 0 ? total / Math.max(1, getMonthsInRange()) : 0;

    // Find top platform
    const platformTotals = {};
    payments.forEach(p => {
        if (!platformTotals[p.platform]) platformTotals[p.platform] = 0;
        platformTotals[p.platform] += parseFloat(p.amount) || 0;
    });
    let topPlatform = '-';
    let maxAmount = 0;
    Object.entries(platformTotals).forEach(([platform, amount]) => {
        if (amount > maxAmount) {
            maxAmount = amount;
            topPlatform = platform;
        }
    });

    document.getElementById('streaming-total').textContent = formatCurrency(total);
    document.getElementById('streaming-monthly-avg').textContent = formatCurrency(monthlyAvg);
    document.getElementById('streaming-top-platform').textContent = topPlatform;

    if (payments.length === 0) {
        container.innerHTML = `
            <div class="empty-state">
                <div class="empty-icon">🎧</div>
                <h3>No streaming payments yet</h3>
                <p>Log your streaming and royalty payments</p>
                <button onclick="openModal('streaming')">+ Add Payment</button>
            </div>
        `;
        return;
    }

    container.innerHTML = payments.map(p => `
        <div class="data-item">
            <div class="data-icon">🎧</div>
            <div class="data-main">
                <div class="data-title">${p.platform}</div>
                <div class="data-subtitle">${p.period || 'Payment'}</div>
            </div>
            <div class="data-meta">
                <div class="data-amount income">+${formatCurrency(p.amount)}</div>
                <div class="data-date">${formatDate(p.date)}</div>
            </div>
            <div class="data-actions">
                <button class="data-btn delete" onclick="confirmDelete('streaming', '${p.id}')">🗑️</button>
            </div>
        </div>
    `).join('');
}

function saveStreaming(e) {
    e.preventDefault();

    const id = document.getElementById('streaming-id').value || generateId();
    const payment = {
        id,
        date: document.getElementById('streaming-date').value,
        platform: document.getElementById('streaming-platform').value,
        amount: document.getElementById('streaming-amount').value,
        period: document.getElementById('streaming-period').value,
        notes: document.getElementById('streaming-notes').value
    };

    appData.streaming.push(payment);
    saveData();
    closeModal('streaming');
    renderStreaming();
    updateDashboard();
    updateCharts();
    showToast('Payment added!');
}

// ========== STUDIO ==========
function renderStudio() {
    const container = document.getElementById('studio-list');
    const sessions = appData.studio.filter(s => isInDateRange(s.date)).sort((a, b) => new Date(b.date) - new Date(a.date));

    const totalCost = sessions.reduce((sum, s) => sum + (parseFloat(s.total) || 0), 0);
    const totalHours = sessions.reduce((sum, s) => sum + (parseFloat(s.hours) || 0), 0);
    const avgRate = totalHours > 0 ? totalCost / totalHours : 0;

    // Count unique projects
    const projects = new Set(sessions.map(s => s.project).filter(Boolean));

    document.getElementById('studio-total-cost').textContent = formatCurrency(totalCost);
    document.getElementById('studio-total-hours').textContent = totalHours;
    document.getElementById('studio-avg-rate').textContent = formatCurrency(avgRate);
    document.getElementById('studio-projects-count').textContent = projects.size;

    if (sessions.length === 0) {
        container.innerHTML = `
            <div class="empty-state">
                <div class="empty-icon">🎚️</div>
                <h3>No studio sessions yet</h3>
                <p>Track your recording and rehearsal time</p>
                <button onclick="openModal('studio')">+ Add Session</button>
            </div>
        `;
        return;
    }

    container.innerHTML = sessions.map(s => `
        <div class="data-item">
            <div class="data-icon">🎚️</div>
            <div class="data-main">
                <div class="data-title">${s.name || s.type}</div>
                <div class="data-subtitle">${s.hours}h @ ${formatCurrency(s.rate)}/hr • ${s.project || 'General'}</div>
            </div>
            <div class="data-meta">
                <div class="data-amount expense">-${formatCurrency(s.total)}</div>
                <div class="data-date">${formatDate(s.date)}</div>
            </div>
            <div class="data-actions">
                <button class="data-btn" onclick="editStudio('${s.id}')">✏️</button>
                <button class="data-btn delete" onclick="confirmDelete('studio', '${s.id}')">🗑️</button>
            </div>
        </div>
    `).join('');
}

function saveStudio(e) {
    e.preventDefault();

    const id = document.getElementById('studio-id').value || generateId();
    const session = {
        id,
        date: document.getElementById('studio-date').value,
        name: document.getElementById('studio-name').value,
        type: document.getElementById('studio-type').value,
        project: document.getElementById('studio-project').value,
        hours: document.getElementById('studio-hours').value,
        rate: document.getElementById('studio-rate').value,
        total: document.getElementById('studio-total').value,
        engineer: document.getElementById('studio-engineer').value,
        notes: document.getElementById('studio-notes').value
    };

    // Calculate total if not set
    if (!session.total && session.hours && session.rate) {
        session.total = parseFloat(session.hours) * parseFloat(session.rate);
    }

    const existingIndex = appData.studio.findIndex(s => s.id === id);
    if (existingIndex >= 0) {
        appData.studio[existingIndex] = session;
    } else {
        appData.studio.push(session);
    }

    saveData();
    closeModal('studio');
    renderStudio();
    updateDashboard();
    updateCharts();
    showToast('Studio session saved!');
}

function editStudio(id) {
    const session = appData.studio.find(s => s.id === id);
    if (!session) return;

    document.getElementById('studio-modal-title').textContent = 'Edit Studio Session';
    document.getElementById('studio-id').value = session.id;
    document.getElementById('studio-date').value = session.date;
    document.getElementById('studio-name').value = session.name || '';
    document.getElementById('studio-type').value = session.type;
    document.getElementById('studio-project').value = session.project || '';
    document.getElementById('studio-hours').value = session.hours;
    document.getElementById('studio-rate').value = session.rate || '';
    document.getElementById('studio-total').value = session.total || '';
    document.getElementById('studio-engineer').value = session.engineer || '';
    document.getElementById('studio-notes').value = session.notes || '';

    openModal('studio');
}

// ========== EQUIPMENT ==========
function renderEquipment() {
    const container = document.getElementById('equipment-list');
    const items = appData.equipment.filter(e => isInDateRange(e.date)).sort((a, b) => new Date(b.date) - new Date(a.date));

    const purchases = items.filter(e => e.type !== 'maintenance' && e.type !== 'rental' && e.type !== 'sale');
    const maintenance = items.filter(e => e.type === 'maintenance' || e.type === 'rental');

    const totalSpent = purchases.reduce((sum, e) => sum + (parseFloat(e.cost) || 0), 0);
    const maintenanceCost = maintenance.reduce((sum, e) => sum + (parseFloat(e.cost) || 0), 0);
    const itemCount = appData.equipment.filter(e => e.type === 'purchase' || e.type === 'used').length;

    document.getElementById('equipment-total-spent').textContent = formatCurrency(totalSpent);
    document.getElementById('equipment-maintenance').textContent = formatCurrency(maintenanceCost);
    document.getElementById('equipment-items-count').textContent = itemCount;

    if (items.length === 0) {
        container.innerHTML = `
            <div class="empty-state">
                <div class="empty-icon">🎸</div>
                <h3>No equipment logged yet</h3>
                <p>Track your gear purchases and maintenance</p>
                <button onclick="openModal('equipment')">+ Add Item</button>
            </div>
        `;
        return;
    }

    container.innerHTML = items.map(e => `
        <div class="data-item">
            <div class="data-icon">🎸</div>
            <div class="data-main">
                <div class="data-title">${e.name}</div>
                <div class="data-subtitle">${e.category} • ${e.type}</div>
            </div>
            <div class="data-meta">
                <div class="data-amount ${e.type === 'sale' ? 'income' : 'expense'}">${e.type === 'sale' ? '+' : '-'}${formatCurrency(e.cost)}</div>
                <div class="data-date">${formatDate(e.date)}</div>
            </div>
            <div class="data-actions">
                <button class="data-btn" onclick="editEquipment('${e.id}')">✏️</button>
                <button class="data-btn delete" onclick="confirmDelete('equipment', '${e.id}')">🗑️</button>
            </div>
        </div>
    `).join('');
}

function saveEquipment(e) {
    e.preventDefault();

    const id = document.getElementById('equipment-id').value || generateId();
    const item = {
        id,
        name: document.getElementById('equipment-name').value,
        category: document.getElementById('equipment-category').value,
        date: document.getElementById('equipment-date').value,
        type: document.getElementById('equipment-type').value,
        cost: document.getElementById('equipment-cost').value,
        value: document.getElementById('equipment-value').value,
        serial: document.getElementById('equipment-serial').value,
        owner: document.getElementById('equipment-owner').value,
        notes: document.getElementById('equipment-notes').value
    };

    const existingIndex = appData.equipment.findIndex(e => e.id === id);
    if (existingIndex >= 0) {
        appData.equipment[existingIndex] = item;
    } else {
        appData.equipment.push(item);
    }

    saveData();
    closeModal('equipment');
    renderEquipment();
    updateDashboard();
    updateCharts();
    showToast('Equipment saved!');
}

function editEquipment(id) {
    const item = appData.equipment.find(e => e.id === id);
    if (!item) return;

    document.getElementById('equipment-modal-title').textContent = 'Edit Equipment';
    document.getElementById('equipment-id').value = item.id;
    document.getElementById('equipment-name').value = item.name;
    document.getElementById('equipment-category').value = item.category;
    document.getElementById('equipment-date').value = item.date;
    document.getElementById('equipment-type').value = item.type;
    document.getElementById('equipment-cost').value = item.cost || '';
    document.getElementById('equipment-value').value = item.value || '';
    document.getElementById('equipment-serial').value = item.serial || '';
    document.getElementById('equipment-owner').value = item.owner;
    document.getElementById('equipment-notes').value = item.notes || '';

    openModal('equipment');
}

// ========== TRAVEL ==========
function renderTravel() {
    const container = document.getElementById('travel-list');
    const trips = appData.travel.filter(t => isInDateRange(t.date)).sort((a, b) => new Date(b.date) - new Date(a.date));

    const totalCost = trips.reduce((sum, t) => sum + (parseFloat(t.cost) || 0), 0);
    const totalDistance = trips.reduce((sum, t) => sum + (parseFloat(t.distance) || 0), 0);
    const fuelCost = trips.filter(t => t.type === 'fuel').reduce((sum, t) => sum + (parseFloat(t.cost) || 0), 0);
    const accommodationCost = trips.filter(t => t.type === 'accommodation').reduce((sum, t) => sum + (parseFloat(t.cost) || 0), 0);

    document.getElementById('travel-total-cost').textContent = formatCurrency(totalCost);
    document.getElementById('travel-total-distance').textContent = formatDistance(totalDistance);
    document.getElementById('travel-fuel-cost').textContent = formatCurrency(fuelCost);
    document.getElementById('travel-accommodation').textContent = formatCurrency(accommodationCost);

    if (trips.length === 0) {
        container.innerHTML = `
            <div class="empty-state">
                <div class="empty-icon">🚐</div>
                <h3>No travel expenses yet</h3>
                <p>Track your tour and travel costs</p>
                <button onclick="openModal('travel')">+ Add Trip</button>
            </div>
        `;
        return;
    }

    container.innerHTML = trips.map(t => `
        <div class="data-item">
            <div class="data-icon">🚐</div>
            <div class="data-main">
                <div class="data-title">${t.type.charAt(0).toUpperCase() + t.type.slice(1)}</div>
                <div class="data-subtitle">${t.from ? `${t.from} → ${t.to}` : t.notes || ''}</div>
            </div>
            <div class="data-meta">
                <div class="data-amount expense">-${formatCurrency(t.cost)}</div>
                <div class="data-date">${formatDate(t.date)}</div>
            </div>
            <div class="data-actions">
                <button class="data-btn delete" onclick="confirmDelete('travel', '${t.id}')">🗑️</button>
            </div>
        </div>
    `).join('');
}

function saveTravel(e) {
    e.preventDefault();

    const id = document.getElementById('travel-id').value || generateId();
    const trip = {
        id,
        date: document.getElementById('travel-date').value,
        type: document.getElementById('travel-type').value,
        from: document.getElementById('travel-from').value,
        to: document.getElementById('travel-to').value,
        distance: document.getElementById('travel-distance').value,
        cost: document.getElementById('travel-cost').value,
        showId: document.getElementById('travel-show').value,
        notes: document.getElementById('travel-notes').value
    };

    appData.travel.push(trip);
    saveData();
    closeModal('travel');
    renderTravel();
    updateDashboard();
    updateCharts();
    showToast('Travel expense saved!');
}

// ========== MARKETING ==========
function renderMarketing() {
    const container = document.getElementById('marketing-list');
    const expenses = appData.marketing.filter(m => isInDateRange(m.date)).sort((a, b) => new Date(b.date) - new Date(a.date));

    // Update category totals
    const categories = {
        social: 'marketing-social',
        design: 'marketing-design',
        pr: 'marketing-pr',
        video: 'marketing-video',
        photo: 'marketing-photo',
        web: 'marketing-web'
    };

    Object.entries(categories).forEach(([cat, elId]) => {
        const total = expenses.filter(e => e.category === cat).reduce((sum, e) => sum + (parseFloat(e.cost) || 0), 0);
        document.getElementById(elId).textContent = formatCurrency(total);
    });

    if (expenses.length === 0) {
        container.innerHTML = `
            <div class="empty-state">
                <div class="empty-icon">📢</div>
                <h3>No marketing expenses yet</h3>
                <p>Track your promotional spending</p>
                <button onclick="openModal('marketing')">+ Add Expense</button>
            </div>
        `;
        return;
    }

    container.innerHTML = expenses.map(m => `
        <div class="data-item">
            <div class="data-icon">📢</div>
            <div class="data-main">
                <div class="data-title">${m.description}</div>
                <div class="data-subtitle">${m.category} • ${m.vendor || ''}</div>
            </div>
            <div class="data-meta">
                <div class="data-amount expense">-${formatCurrency(m.cost)}</div>
                <div class="data-date">${formatDate(m.date)}</div>
            </div>
            <div class="data-actions">
                <button class="data-btn delete" onclick="confirmDelete('marketing', '${m.id}')">🗑️</button>
            </div>
        </div>
    `).join('');
}

function saveMarketing(e) {
    e.preventDefault();

    const id = document.getElementById('marketing-id').value || generateId();
    const expense = {
        id,
        date: document.getElementById('marketing-date').value,
        category: document.getElementById('marketing-category').value,
        description: document.getElementById('marketing-description').value,
        vendor: document.getElementById('marketing-vendor').value,
        cost: document.getElementById('marketing-cost').value,
        release: document.getElementById('marketing-release').value,
        notes: document.getElementById('marketing-notes').value
    };

    appData.marketing.push(expense);
    saveData();
    closeModal('marketing');
    renderMarketing();
    updateDashboard();
    updateCharts();
    showToast('Marketing expense saved!');
}

// ========== OTHER INCOME ==========
function renderOtherIncome() {
    const container = document.getElementById('other-income-list');
    const income = appData.otherIncome.filter(i => isInDateRange(i.date)).sort((a, b) => new Date(b.date) - new Date(a.date));

    // Update category totals
    const categories = {
        lessons: 'income-lessons',
        sync: 'income-sync',
        sponsors: 'income-sponsors',
        crowdfund: 'income-crowdfund',
        session: 'income-session',
        other: 'income-other'
    };

    Object.entries(categories).forEach(([cat, elId]) => {
        const total = income.filter(i => i.category === cat).reduce((sum, i) => sum + (parseFloat(i.amount) || 0), 0);
        document.getElementById(elId).textContent = formatCurrency(total);
    });

    if (income.length === 0) {
        container.innerHTML = `
            <div class="empty-state">
                <div class="empty-icon">💵</div>
                <h3>No other income yet</h3>
                <p>Track additional revenue streams</p>
                <button onclick="openModal('other-income')">+ Add Income</button>
            </div>
        `;
        return;
    }

    container.innerHTML = income.map(i => `
        <div class="data-item">
            <div class="data-icon">💵</div>
            <div class="data-main">
                <div class="data-title">${i.description}</div>
                <div class="data-subtitle">${i.category}</div>
            </div>
            <div class="data-meta">
                <div class="data-amount income">+${formatCurrency(i.amount)}</div>
                <div class="data-date">${formatDate(i.date)}</div>
            </div>
            <div class="data-actions">
                <button class="data-btn delete" onclick="confirmDelete('other-income', '${i.id}')">🗑️</button>
            </div>
        </div>
    `).join('');
}

function saveOtherIncome(e) {
    e.preventDefault();

    const id = document.getElementById('other-income-id').value || generateId();
    const income = {
        id,
        date: document.getElementById('other-income-date').value,
        category: document.getElementById('other-income-category').value,
        description: document.getElementById('other-income-description').value,
        amount: document.getElementById('other-income-amount').value,
        notes: document.getElementById('other-income-notes').value
    };

    appData.otherIncome.push(income);
    saveData();
    closeModal('other-income');
    renderOtherIncome();
    updateDashboard();
    showToast('Income added!');
}

// ========== OTHER EXPENSES ==========
function renderOtherExpenses() {
    const container = document.getElementById('other-expenses-list');
    const expenses = appData.otherExpenses.filter(e => isInDateRange(e.date)).sort((a, b) => new Date(b.date) - new Date(a.date));

    // Update category totals
    const categories = {
        insurance: 'expense-insurance',
        legal: 'expense-legal',
        bank: 'expense-bank',
        storage: 'expense-storage',
        memberships: 'expense-memberships',
        misc: 'expense-misc'
    };

    Object.entries(categories).forEach(([cat, elId]) => {
        const total = expenses.filter(e => e.category === cat).reduce((sum, e) => sum + (parseFloat(e.amount) || 0), 0);
        document.getElementById(elId).textContent = formatCurrency(total);
    });

    if (expenses.length === 0) {
        container.innerHTML = `
            <div class="empty-state">
                <div class="empty-icon">📋</div>
                <h3>No other expenses yet</h3>
                <p>Track miscellaneous costs</p>
                <button onclick="openModal('other-expense')">+ Add Expense</button>
            </div>
        `;
        return;
    }

    container.innerHTML = expenses.map(e => `
        <div class="data-item">
            <div class="data-icon">📋</div>
            <div class="data-main">
                <div class="data-title">${e.description}</div>
                <div class="data-subtitle">${e.category} • ${e.vendor || ''}</div>
            </div>
            <div class="data-meta">
                <div class="data-amount expense">-${formatCurrency(e.amount)}</div>
                <div class="data-date">${formatDate(e.date)}</div>
            </div>
            <div class="data-actions">
                <button class="data-btn delete" onclick="confirmDelete('other-expense', '${e.id}')">🗑️</button>
            </div>
        </div>
    `).join('');
}

function saveOtherExpense(e) {
    e.preventDefault();

    const id = document.getElementById('other-expense-id').value || generateId();
    const expense = {
        id,
        date: document.getElementById('other-expense-date').value,
        category: document.getElementById('other-expense-category').value,
        description: document.getElementById('other-expense-description').value,
        amount: document.getElementById('other-expense-amount').value,
        vendor: document.getElementById('other-expense-vendor').value,
        notes: document.getElementById('other-expense-notes').value
    };

    appData.otherExpenses.push(expense);
    saveData();
    closeModal('other-expense');
    renderOtherExpenses();
    updateDashboard();
    updateCharts();
    showToast('Expense added!');
}

// ========== TAX SUMMARY ==========
function updateTaxSummary() {
    const year = parseInt(document.getElementById('tax-year').value) || new Date().getFullYear();

    // Filter for tax year
    const isInTaxYear = (dateStr) => {
        const date = new Date(dateStr);
        return date.getFullYear() === year;
    };

    // Income
    const showFees = appData.shows.filter(s => isInTaxYear(s.date)).reduce((sum, s) => sum + (parseFloat(s.fee) || 0) + (parseFloat(s.door) || 0), 0);
    const merchSales = appData.merchSales.filter(s => isInTaxYear(s.date)).reduce((sum, s) => sum + (parseFloat(s.total) || 0), 0);
    const streaming = appData.streaming.filter(s => isInTaxYear(s.date)).reduce((sum, s) => sum + (parseFloat(s.amount) || 0), 0);
    const otherIncome = appData.otherIncome.filter(i => isInTaxYear(i.date)).reduce((sum, i) => sum + (parseFloat(i.amount) || 0), 0);
    const totalIncome = showFees + merchSales + streaming + otherIncome;

    // Expenses
    const crewCost = appData.shows.filter(s => isInTaxYear(s.date)).reduce((sum, s) => sum + (parseFloat(s.crewCost) || 0), 0);
    const studioCost = appData.studio.filter(s => isInTaxYear(s.date)).reduce((sum, s) => sum + (parseFloat(s.total) || 0), 0);
    const equipmentCost = appData.equipment.filter(e => isInTaxYear(e.date) && e.type !== 'sale').reduce((sum, e) => sum + (parseFloat(e.cost) || 0), 0);
    const travelCost = appData.travel.filter(t => isInTaxYear(t.date)).reduce((sum, t) => sum + (parseFloat(t.cost) || 0), 0);
    const marketingCost = appData.marketing.filter(m => isInTaxYear(m.date)).reduce((sum, m) => sum + (parseFloat(m.cost) || 0), 0);
    const merchCost = appData.merchOrders.filter(o => isInTaxYear(o.date)).reduce((sum, o) => sum + (parseFloat(o.totalCost) || 0), 0);
    const otherExpenses = appData.otherExpenses.filter(e => isInTaxYear(e.date)).reduce((sum, e) => sum + (parseFloat(e.amount) || 0), 0);
    const totalDeductions = crewCost + studioCost + equipmentCost + travelCost + marketingCost + merchCost + otherExpenses;

    // Tax calculation
    const taxableIncome = Math.max(0, totalIncome - totalDeductions);
    const taxRate = parseFloat(appData.settings.incomeTaxRate) || 0;
    const estimatedTax = taxableIncome * (taxRate / 100);

    // Update display
    document.getElementById('tax-show-fees').textContent = formatCurrency(showFees);
    document.getElementById('tax-merch-sales').textContent = formatCurrency(merchSales);
    document.getElementById('tax-streaming').textContent = formatCurrency(streaming);
    document.getElementById('tax-other-income').textContent = formatCurrency(otherIncome);
    document.getElementById('tax-total-income').textContent = formatCurrency(totalIncome);

    document.getElementById('tax-crew').textContent = formatCurrency(crewCost);
    document.getElementById('tax-studio').textContent = formatCurrency(studioCost);
    document.getElementById('tax-equipment').textContent = formatCurrency(equipmentCost);
    document.getElementById('tax-travel').textContent = formatCurrency(travelCost);
    document.getElementById('tax-marketing').textContent = formatCurrency(marketingCost);
    document.getElementById('tax-merch-cost').textContent = formatCurrency(merchCost);
    document.getElementById('tax-other-expenses').textContent = formatCurrency(otherExpenses);
    document.getElementById('tax-total-deductions').textContent = formatCurrency(totalDeductions);

    document.getElementById('tax-calc-gross').textContent = formatCurrency(totalIncome);
    document.getElementById('tax-calc-deductions').textContent = formatCurrency(totalDeductions);
    document.getElementById('tax-calc-taxable').textContent = formatCurrency(taxableIncome);
    document.getElementById('tax-calc-rate').textContent = `${taxRate}%`;
    document.getElementById('tax-calc-amount').textContent = formatCurrency(estimatedTax);
}

// ========== REPORTS ==========
function generateReport() {
    const container = document.getElementById('report-content');
    const { start, end } = getDateRange();

    // Gather all data
    let totalIncome = 0, totalExpenses = 0;

    // Income breakdown
    const showIncome = appData.shows.filter(s => isInDateRange(s.date)).reduce((sum, s) => sum + (parseFloat(s.fee) || 0) + (parseFloat(s.merch) || 0) + (parseFloat(s.door) || 0), 0);
    const merchIncome = appData.merchSales.filter(s => isInDateRange(s.date)).reduce((sum, s) => sum + (parseFloat(s.total) || 0), 0);
    const streamingIncome = appData.streaming.filter(s => isInDateRange(s.date)).reduce((sum, s) => sum + (parseFloat(s.amount) || 0), 0);
    const otherIncomeTotal = appData.otherIncome.filter(i => isInDateRange(i.date)).reduce((sum, i) => sum + (parseFloat(i.amount) || 0), 0);
    totalIncome = showIncome + merchIncome + streamingIncome + otherIncomeTotal;

    // Expense breakdown
    const showCosts = appData.shows.filter(s => isInDateRange(s.date)).reduce((sum, s) => sum + (parseFloat(s.travelCost) || 0) + (parseFloat(s.accommodation) || 0) + (parseFloat(s.crewCost) || 0) + (parseFloat(s.otherCost) || 0), 0);
    const merchCosts = appData.merchOrders.filter(o => isInDateRange(o.date)).reduce((sum, o) => sum + (parseFloat(o.totalCost) || 0), 0);
    const studioCosts = appData.studio.filter(s => isInDateRange(s.date)).reduce((sum, s) => sum + (parseFloat(s.total) || 0), 0);
    const equipmentCosts = appData.equipment.filter(e => isInDateRange(e.date) && e.type !== 'sale').reduce((sum, e) => sum + (parseFloat(e.cost) || 0), 0);
    const travelCosts = appData.travel.filter(t => isInDateRange(t.date)).reduce((sum, t) => sum + (parseFloat(t.cost) || 0), 0);
    const marketingCosts = appData.marketing.filter(m => isInDateRange(m.date)).reduce((sum, m) => sum + (parseFloat(m.cost) || 0), 0);
    const otherCosts = appData.otherExpenses.filter(e => isInDateRange(e.date)).reduce((sum, e) => sum + (parseFloat(e.amount) || 0), 0);
    totalExpenses = showCosts + merchCosts + studioCosts + equipmentCosts + travelCosts + marketingCosts + otherCosts;

    const netProfit = totalIncome - totalExpenses;

    container.innerHTML = `
        <div class="report-header">
            <h2>Financial Report - ${appData.settings.bandName}</h2>
            <div class="report-period">${formatDate(start)} to ${formatDate(end)}</div>
        </div>

        <div class="report-section">
            <h3>Income</h3>
            <div class="report-row">
                <span class="label">Show Fees & Door</span>
                <span class="value">${formatCurrency(showIncome)}</span>
            </div>
            <div class="report-row">
                <span class="label">Merch Sales</span>
                <span class="value">${formatCurrency(merchIncome)}</span>
            </div>
            <div class="report-row">
                <span class="label">Streaming & Royalties</span>
                <span class="value">${formatCurrency(streamingIncome)}</span>
            </div>
            <div class="report-row">
                <span class="label">Other Income</span>
                <span class="value">${formatCurrency(otherIncomeTotal)}</span>
            </div>
            <div class="report-row total">
                <span class="label">Total Income</span>
                <span class="value">${formatCurrency(totalIncome)}</span>
            </div>
        </div>

        <div class="report-section">
            <h3>Expenses</h3>
            <div class="report-row">
                <span class="label">Show Costs (Travel, Crew, etc.)</span>
                <span class="value">${formatCurrency(showCosts)}</span>
            </div>
            <div class="report-row">
                <span class="label">Merch Production</span>
                <span class="value">${formatCurrency(merchCosts)}</span>
            </div>
            <div class="report-row">
                <span class="label">Studio & Recording</span>
                <span class="value">${formatCurrency(studioCosts)}</span>
            </div>
            <div class="report-row">
                <span class="label">Equipment & Gear</span>
                <span class="value">${formatCurrency(equipmentCosts)}</span>
            </div>
            <div class="report-row">
                <span class="label">Travel & Transport</span>
                <span class="value">${formatCurrency(travelCosts)}</span>
            </div>
            <div class="report-row">
                <span class="label">Marketing & Promo</span>
                <span class="value">${formatCurrency(marketingCosts)}</span>
            </div>
            <div class="report-row">
                <span class="label">Other Expenses</span>
                <span class="value">${formatCurrency(otherCosts)}</span>
            </div>
            <div class="report-row total">
                <span class="label">Total Expenses</span>
                <span class="value">${formatCurrency(totalExpenses)}</span>
            </div>
        </div>

        <div class="report-section">
            <h3>Summary</h3>
            <div class="report-row">
                <span class="label">Total Income</span>
                <span class="value" style="color: var(--success-green)">${formatCurrency(totalIncome)}</span>
            </div>
            <div class="report-row">
                <span class="label">Total Expenses</span>
                <span class="value" style="color: var(--error-red)">${formatCurrency(totalExpenses)}</span>
            </div>
            <div class="report-row total">
                <span class="label">Net Profit/Loss</span>
                <span class="value" style="color: ${netProfit >= 0 ? 'var(--success-green)' : 'var(--error-red)'}">${formatCurrency(netProfit)}</span>
            </div>
        </div>
    `;
}

// ========== SETTINGS ==========
function saveSettings() {
    appData.settings.bandName = document.getElementById('settings-band-name').value || 'My Band';
    appData.settings.genre = document.getElementById('settings-genre').value;
    appData.settings.formationYear = document.getElementById('settings-year').value;
    appData.settings.currency = document.getElementById('settings-currency').value;
    appData.settings.distanceUnit = document.getElementById('settings-distance').value;
    appData.settings.dateFormat = document.getElementById('settings-date-format').value;
    appData.settings.incomeTaxRate = document.getElementById('settings-income-tax').value;
    appData.settings.salesTaxRate = document.getElementById('settings-sales-tax').value;
    appData.settings.taxYearStart = document.getElementById('settings-tax-year-start').value;
    appData.settings.vatRegistered = document.getElementById('settings-vat-registered').checked;
    appData.settings.defaultFee = document.getElementById('settings-default-fee').value;
    appData.settings.mileageRate = document.getElementById('settings-mileage-rate').value;
    appData.settings.perDiem = document.getElementById('settings-per-diem').value;
    appData.settings.theme = document.getElementById('settings-theme').value;

    saveData();
    applySettings();
    showToast('Settings saved!');
}

// ========== BACKUP & EXPORT ==========
function createBackup() {
    const backup = {
        version: '1.0',
        created: new Date().toISOString(),
        data: appData
    };

    const blob = new Blob([JSON.stringify(backup, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `pingig-finance-backup-${formatDate(new Date())}.json`;
    a.click();
    URL.revokeObjectURL(url);

    appData.lastBackup = new Date().toISOString();
    saveData();
    document.getElementById('last-backup').textContent = `Last backup: ${formatDate(appData.lastBackup)}`;

    showToast('Backup created!');
}

function restoreBackup(input) {
    const file = input.files[0];
    if (!file) return;

    const reader = new FileReader();
    reader.onload = (e) => {
        try {
            const backup = JSON.parse(e.target.result);
            if (backup.data) {
                appData = { ...appData, ...backup.data };
                saveData();
                applySettings();
                loadViewData(currentView);
                updateCharts();
                showToast('Backup restored successfully!', 'success');
            } else {
                showToast('Invalid backup file', 'error');
            }
        } catch (err) {
            showToast('Error reading backup file', 'error');
        }
    };
    reader.readAsText(file);
}

function exportCSV(type) {
    let csv = '';
    let filename = '';

    switch(type) {
        case 'all':
            // Export comprehensive summary
            csv = 'Category,Type,Date,Description,Amount\n';

            appData.shows.forEach(s => {
                csv += `Shows,Income,${s.date},"${s.venue} - ${s.city}",${(parseFloat(s.fee) || 0) + (parseFloat(s.merch) || 0)}\n`;
            });
            appData.streaming.forEach(s => {
                csv += `Streaming,Income,${s.date},"${s.platform}",${s.amount}\n`;
            });
            appData.studio.forEach(s => {
                csv += `Studio,Expense,${s.date},"${s.name || s.type}",${s.total}\n`;
            });
            appData.equipment.forEach(e => {
                csv += `Equipment,${e.type === 'sale' ? 'Income' : 'Expense'},${e.date},"${e.name}",${e.cost}\n`;
            });

            filename = 'pingig-all-data.csv';
            break;

        case 'shows':
            csv = 'Date,Venue,City,Type,Fee,Merch,Door,Travel Cost,Accommodation,Crew Cost,Other Cost,Attendance\n';
            appData.shows.forEach(s => {
                csv += `${s.date},"${s.venue}","${s.city}",${s.type},${s.fee || 0},${s.merch || 0},${s.door || 0},${s.travelCost || 0},${s.accommodation || 0},${s.crewCost || 0},${s.otherCost || 0},${s.attendance || 0}\n`;
            });
            filename = 'pingig-shows.csv';
            break;

        case 'expenses':
            csv = 'Date,Category,Description,Amount\n';
            appData.studio.forEach(s => csv += `${s.date},Studio,"${s.name || s.type}",${s.total}\n`);
            appData.equipment.filter(e => e.type !== 'sale').forEach(e => csv += `${e.date},Equipment,"${e.name}",${e.cost}\n`);
            appData.travel.forEach(t => csv += `${t.date},Travel,"${t.type}",${t.cost}\n`);
            appData.marketing.forEach(m => csv += `${m.date},Marketing,"${m.description}",${m.cost}\n`);
            appData.otherExpenses.forEach(e => csv += `${e.date},Other,"${e.description}",${e.amount}\n`);
            filename = 'pingig-expenses.csv';
            break;

        case 'income':
            csv = 'Date,Source,Description,Amount\n';
            appData.shows.forEach(s => csv += `${s.date},Show,"${s.venue}",${(parseFloat(s.fee) || 0) + (parseFloat(s.merch) || 0)}\n`);
            appData.merchSales.forEach(s => csv += `${s.date},Merch Sale,"${s.location}",${s.total}\n`);
            appData.streaming.forEach(s => csv += `${s.date},Streaming,"${s.platform}",${s.amount}\n`);
            appData.otherIncome.forEach(i => csv += `${i.date},Other,"${i.description}",${i.amount}\n`);
            filename = 'pingig-income.csv';
            break;
    }

    const blob = new Blob([csv], { type: 'text/csv' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = filename;
    a.click();
    URL.revokeObjectURL(url);

    showToast('CSV exported!');
}

function confirmReset() {
    if (confirm('Are you ABSOLUTELY SURE you want to delete ALL your data? This cannot be undone!')) {
        if (confirm('This is your last chance. All shows, expenses, crew data, EVERYTHING will be deleted. Continue?')) {
            localStorage.removeItem('pingigFinanceData');
            location.reload();
        }
    }
}

// ========== MODALS ==========
function openModal(type) {
    const modal = document.getElementById(`${type}-modal`);
    if (modal) {
        modal.classList.add('active');

        // Reset form if it exists
        const form = modal.querySelector('form');
        if (form) {
            form.reset();
            // Clear hidden ID field
            const idField = form.querySelector('input[type="hidden"]');
            if (idField) idField.value = '';
        }

        // Set default date to today
        const dateField = modal.querySelector('input[type="date"]');
        if (dateField && !dateField.value) {
            dateField.value = new Date().toISOString().split('T')[0];
        }

        // Reset modal titles
        const titles = {
            'show': 'Add Show',
            'crew': 'Add Crew Member',
            'studio': 'Add Studio Session',
            'equipment': 'Add Equipment',
            'merch-order': 'Add Production Order'
        };
        const titleEl = modal.querySelector('.modal-header h2');
        if (titleEl && titles[type]) {
            titleEl.textContent = titles[type];
        }

        // Populate dropdowns
        if (type === 'merch-sale') {
            populateMerchProductDropdown();
            populateShowDropdown('merch-sale-show');
        }
        if (type === 'travel') {
            populateShowDropdown('travel-show');
        }
        if (type === 'equipment') {
            populateCrewDropdown();
        }
    }
}

function closeModal(type) {
    const modal = document.getElementById(`${type}-modal`);
    if (modal) {
        modal.classList.remove('active');
    }
}

function openQuickAdd() {
    openModal('quick-add');
}

function populateMerchProductDropdown() {
    const select = document.getElementById('merch-sale-product');
    select.innerHTML = '<option value="">Select product...</option>';
    appData.merchProducts.forEach(p => {
        select.innerHTML += `<option value="${p.id}">${p.name} (${formatCurrency(p.sellPrice)})</option>`;
    });
}

function populateShowDropdown(selectId) {
    const select = document.getElementById(selectId);
    if (!select) return;
    select.innerHTML = '<option value="">Not related to a show</option>';
    appData.shows.sort((a, b) => new Date(b.date) - new Date(a.date)).slice(0, 20).forEach(s => {
        select.innerHTML += `<option value="${s.id}">${formatDate(s.date)} - ${s.venue}</option>`;
    });
}

function populateCrewDropdown() {
    const select = document.getElementById('equipment-owner');
    if (!select) return;
    select.innerHTML = '<option value="band">Band (Shared)</option>';
    appData.crew.forEach(c => {
        select.innerHTML += `<option value="${c.id}">${c.name}</option>`;
    });
}

// ========== DELETE CONFIRMATION ==========
let deleteTarget = { type: '', id: '' };

function confirmDelete(type, id) {
    deleteTarget = { type, id };

    const messages = {
        show: 'Are you sure you want to delete this show?',
        crew: 'Are you sure you want to remove this crew member?',
        'merch-sale': 'Are you sure you want to delete this sale record?',
        'merch-order': 'Are you sure you want to delete this production order?',
        streaming: 'Are you sure you want to delete this payment?',
        studio: 'Are you sure you want to delete this studio session?',
        equipment: 'Are you sure you want to delete this equipment record?',
        travel: 'Are you sure you want to delete this travel expense?',
        marketing: 'Are you sure you want to delete this marketing expense?',
        'other-income': 'Are you sure you want to delete this income?',
        'other-expense': 'Are you sure you want to delete this expense?'
    };

    document.getElementById('confirm-message').textContent = messages[type] || 'Are you sure you want to delete this item?';
    document.getElementById('confirm-delete-btn').onclick = executeDelete;
    openModal('confirm');
}

function executeDelete() {
    const { type, id } = deleteTarget;

    switch(type) {
        case 'show': deleteShow(id); break;
        case 'crew': deleteCrew(id); break;
        case 'merch-sale':
            appData.merchSales = appData.merchSales.filter(s => s.id !== id);
            saveData();
            renderMerchSales();
            break;
        case 'merch-order':
            appData.merchOrders = appData.merchOrders.filter(o => o.id !== id);
            saveData();
            renderMerchCosts();
            break;
        case 'streaming':
            appData.streaming = appData.streaming.filter(s => s.id !== id);
            saveData();
            renderStreaming();
            break;
        case 'studio':
            appData.studio = appData.studio.filter(s => s.id !== id);
            saveData();
            renderStudio();
            break;
        case 'equipment':
            appData.equipment = appData.equipment.filter(e => e.id !== id);
            saveData();
            renderEquipment();
            break;
        case 'travel':
            appData.travel = appData.travel.filter(t => t.id !== id);
            saveData();
            renderTravel();
            break;
        case 'marketing':
            appData.marketing = appData.marketing.filter(m => m.id !== id);
            saveData();
            renderMarketing();
            break;
        case 'other-income':
            appData.otherIncome = appData.otherIncome.filter(i => i.id !== id);
            saveData();
            renderOtherIncome();
            break;
        case 'other-expense':
            appData.otherExpenses = appData.otherExpenses.filter(e => e.id !== id);
            saveData();
            renderOtherExpenses();
            break;
    }

    closeModal('confirm');
    updateDashboard();
    updateCharts();
    showToast('Item deleted');
}

// ========== FORMS ==========
function initForms() {
    // Auto-calculate merch sale total
    ['merch-sale-qty', 'merch-sale-price'].forEach(id => {
        const el = document.getElementById(id);
        if (el) {
            el.addEventListener('input', () => {
                const qty = parseFloat(document.getElementById('merch-sale-qty').value) || 0;
                const price = parseFloat(document.getElementById('merch-sale-price').value) || 0;
                document.getElementById('merch-sale-total').value = (qty * price).toFixed(2);
            });
        }
    });

    // Auto-fill price from product
    document.getElementById('merch-sale-product')?.addEventListener('change', (e) => {
        const product = appData.merchProducts.find(p => p.id === e.target.value);
        if (product) {
            document.getElementById('merch-sale-price').value = product.sellPrice || '';
            const qty = parseFloat(document.getElementById('merch-sale-qty').value) || 1;
            document.getElementById('merch-sale-total').value = (qty * (product.sellPrice || 0)).toFixed(2);
        }
    });

    // Auto-calculate merch order total
    ['merch-order-qty', 'merch-cost-per'].forEach(id => {
        const el = document.getElementById(id);
        if (el) {
            el.addEventListener('input', () => {
                const qty = parseFloat(document.getElementById('merch-order-qty').value) || 0;
                const cost = parseFloat(document.getElementById('merch-cost-per').value) || 0;
                document.getElementById('merch-total-cost').value = (qty * cost).toFixed(2);
            });
        }
    });

    // Auto-calculate studio total
    ['studio-hours', 'studio-rate'].forEach(id => {
        const el = document.getElementById(id);
        if (el) {
            el.addEventListener('input', () => {
                const hours = parseFloat(document.getElementById('studio-hours').value) || 0;
                const rate = parseFloat(document.getElementById('studio-rate').value) || 0;
                document.getElementById('studio-total').value = (hours * rate).toFixed(2);
            });
        }
    });

    // Tax year change
    document.getElementById('tax-year')?.addEventListener('change', updateTaxSummary);

    // View tabs
    document.querySelectorAll('.view-tab').forEach(tab => {
        tab.addEventListener('click', () => {
            const parent = tab.closest('.view-header');
            parent.querySelectorAll('.view-tab').forEach(t => t.classList.remove('active'));
            tab.classList.add('active');
            // Could add subtab filtering here
        });
    });

    // Color buttons
    document.querySelectorAll('.color-btn').forEach(btn => {
        btn.addEventListener('click', () => {
            document.querySelectorAll('.color-btn').forEach(b => b.classList.remove('active'));
            btn.classList.add('active');
            appData.settings.accentColor = btn.dataset.color;
        });
    });
}

// ========== UTILITIES ==========
function generateId() {
    return Date.now().toString(36) + Math.random().toString(36).substr(2);
}

function showToast(message, type = 'success') {
    const toast = document.getElementById('toast');
    toast.textContent = message;
    toast.className = `toast ${type} show`;

    setTimeout(() => {
        toast.classList.remove('show');
    }, 3000);
}

function getMonthsInRange() {
    const { start, end } = getDateRange();
    const months = (end.getFullYear() - start.getFullYear()) * 12 + (end.getMonth() - start.getMonth()) + 1;
    return Math.max(1, months);
}

function exportReport() {
    showToast('PDF export coming soon!');
}

function exportTaxReport() {
    // Export as CSV for accountant
    const year = document.getElementById('tax-year').value;
    let csv = `Tax Summary Report - ${appData.settings.bandName} - ${year}\n\n`;
    csv += 'INCOME\n';
    csv += `Show Fees,${document.getElementById('tax-show-fees').textContent}\n`;
    csv += `Merch Sales,${document.getElementById('tax-merch-sales').textContent}\n`;
    csv += `Streaming/Royalties,${document.getElementById('tax-streaming').textContent}\n`;
    csv += `Other Income,${document.getElementById('tax-other-income').textContent}\n`;
    csv += `TOTAL INCOME,${document.getElementById('tax-total-income').textContent}\n\n`;
    csv += 'DEDUCTIONS\n';
    csv += `Crew & Contractors,${document.getElementById('tax-crew').textContent}\n`;
    csv += `Studio & Recording,${document.getElementById('tax-studio').textContent}\n`;
    csv += `Equipment & Gear,${document.getElementById('tax-equipment').textContent}\n`;
    csv += `Travel & Transport,${document.getElementById('tax-travel').textContent}\n`;
    csv += `Marketing & Promo,${document.getElementById('tax-marketing').textContent}\n`;
    csv += `Merch Production,${document.getElementById('tax-merch-cost').textContent}\n`;
    csv += `Other Expenses,${document.getElementById('tax-other-expenses').textContent}\n`;
    csv += `TOTAL DEDUCTIONS,${document.getElementById('tax-total-deductions').textContent}\n\n`;
    csv += 'CALCULATION\n';
    csv += `Taxable Income,${document.getElementById('tax-calc-taxable').textContent}\n`;
    csv += `Tax Rate,${document.getElementById('tax-calc-rate').textContent}\n`;
    csv += `Estimated Tax,${document.getElementById('tax-calc-amount').textContent}\n`;

    const blob = new Blob([csv], { type: 'text/csv' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `tax-summary-${year}.csv`;
    a.click();
    URL.revokeObjectURL(url);

    showToast('Tax report exported!');
}

function addCustomCategory(type) {
    const name = prompt(`Enter new ${type} category name:`);
    if (name) {
        if (type === 'income') {
            appData.settings.customIncomeCategories.push(name);
        } else {
            appData.settings.customExpenseCategories.push(name);
        }
        saveData();
        showToast('Category added!');
    }
}

// Initialize on load
console.log('🎸 PINGIG Finance loaded!');
